/* kintone-dupcheck-plugin: 人材業向け 重複チェック（一覧バッチ検出） */
(function () {
  'use strict';

  kintone.events.on('app.record.index.show', function (event) {
    const header = kintone.app.getHeaderMenuSpaceElement();
    if (!header || header.dataset.dupButtonAttached) return event;

    const btn = document.createElement('button');
    btn.textContent = '重複チェック';
    btn.style.cssText = 'margin-left:8px; padding:8px 12px; border:1px solid #ddd; border-radius:6px; background:#fff; cursor:pointer;';
    btn.onclick = runDuplicateScan;
    header.appendChild(btn);
    header.dataset.dupButtonAttached = '1';
    return event;
  });

  async function runDuplicateScan() {
    const appId = kintone.app.getId();
    const overlay = showOverlay('重複チェックを実行中…');

    try {
      const all = await fetchAllRecords(appId);
      const groups = buildDuplicateGroups(all);
      renderResult(groups);
    } catch (e) {
      alert('重複チェックでエラーが発生しました。コンソールを確認してください。');
      console.error(e);
    } finally {
      hideOverlay(overlay);
    }
  }

  async function fetchAllRecords(appId) {
    const limit = 500;
    let offset = 0;
    const all = [];
    while (true) {
      const query = `order by $id asc limit ${limit} offset ${offset}`;
      const resp = await kintone.api('/k/v1/records', 'GET', { app: appId, query });
      all.push(...resp.records);
      if (resp.records.length < limit) break;
      offset += limit;
    }
    return all;
  }

  function normalizeKana(s = '') {
    return String(s).replace(/\s+/g, '').replace(/[ー−‐-]/g, '').toLowerCase();
  }
  function normalizePhone(s = '') { return String(s).replace(/\D/g, ''); }

  // 簡易バイグラム類似度（Dice係数）: 0〜1
  // 表記揺れ対応：スペース除去、カタカナ統一、類似文字正規化
  function bigramSimilarity(a = '', b = '') {
    // 類似文字の正規化マップ（旧字体、異体字、よくある表記揺れ）
    const similarChars = {
      // 「郎」「朗」系
      '朗': '郎', '廊': '郎',
      // 「沢」「澤」系
      '澤': '沢', '泽': '沢',
      // 「斉」「斎」「齋」系
      '斎': '斉', '齋': '斉', '齊': '斉',
      // 「辺」「邊」系
      '邊': '辺', '邉': '辺',
      // 「高」「髙」系
      '髙': '高', '﨑': '崎',
      // 数字
      '１': '1', '２': '2', '３': '3', '４': '4', '５': '5',
      '６': '6', '７': '7', '８': '8', '９': '9', '０': '0',
      // その他よくある揺れ
      '衞': '衛', '國': '国', '實': '実', '淸': '清',
      '濵': '浜', '濱': '浜', '渕': '淵', '嶋': '島'
    };

    // 前処理関数
    const normalize = str => {
      // 1. 空白・スペースを完全除去
      str = String(str).replace(/[\s　]/g, '');
      // 2. カタカナをひらがなに統一
      str = str.replace(/[\u30A1-\u30F6]/g, c => String.fromCharCode(c.charCodeAt(0) - 0x60));
      // 3. 類似文字を正規化
      str = str.split('').map(c => similarChars[c] || c).join('');
      // 4. 小文字化（英数字）
      str = str.toLowerCase();
      return str;
    };

    a = normalize(a);
    b = normalize(b);

    // 完全一致チェック
    if (a === b) return 1;

    // 長さチェック（正規化後に再度確認）
    if (a.length < 2 || b.length < 2) {
      // 1文字同士の比較は完全一致のみ
      return a === b ? 1 : 0;
    }

    const bigrams = str => new Map(
      Array.from({ length: str.length - 1 }, (_, i) => str.slice(i, i + 2))
        .reduce((m, bg) => m.set(bg, (m.get(bg) || 0) + 1), new Map())
    );
    const A = bigrams(a), B = bigrams(b);
    let overlap = 0;
    for (const [bg, cnt] of A) overlap += Math.min(cnt, B.get(bg) || 0);
    const sizeA = [...A.values()].reduce((s, v) => s + v, 0);
    const sizeB = [...B.values()].reduce((s, v) => s + v, 0);
    return (2 * overlap) / (sizeA + sizeB);
  }

  function buildDuplicateGroups(records) {
    const groups = [];
    const byEmail = new Map();
    const byKanaDob = new Map();

    // 1) 厳密一致: email
    for (const r of records) {
      const email = String(r.email?.value || '').trim().toLowerCase();
      if (!email) continue;
      if (!byEmail.has(email)) byEmail.set(email, []);
      byEmail.get(email).push(r);
    }
    for (const arr of byEmail.values()) if (arr.length > 1) groups.push({ rule: 'email', members: arr });

    // 2) 準一致: name_kana + dob
    for (const r of records) {
      const kana = normalizeKana(r.name_kana?.value || '');
      const dob  = String(r.dob?.value || '');
      if (!kana || !dob) continue;
      const key = `${kana}__${dob}`;
      if (!byKanaDob.has(key)) byKanaDob.set(key, []);
      byKanaDob.get(key).push(r);
    }
    for (const arr of byKanaDob.values()) if (arr.length > 1) groups.push({ rule: 'kana+dob', members: arr });

    // 3) あいまい: name 類似 >= 0.90 && tel 下4桁一致
    const seen = new Set();
    for (let i = 0; i < records.length; i++) {
      for (let j = i + 1; j < records.length; j++) {
        const a = records[i], b = records[j];
        const nameA = (a.name?.value || '').trim();
        const nameB = (b.name?.value || '').trim();
        if (!nameA || !nameB) continue;
        const sim = bigramSimilarity(nameA, nameB);
        if (sim >= 0.90) {
          const tA = normalizePhone(a.tel?.value || '');
          const tB = normalizePhone(b.tel?.value || '');
          const last4A = tA.slice(-4), last4B = tB.slice(-4);
          if (last4A && last4B && last4A === last4B) {
            const key = [a.$id.value, b.$id.value].sort().join('-');
            if (!seen.has(key)) {
              groups.push({ rule: 'fuzzy(name)+tel4', members: [a, b], score: sim });
              seen.add(key);
            }
          }
        }
      }
    }
    return groups;
  }

  function showOverlay(text) {
    const o = document.createElement('div');
    o.style.cssText = 'position:fixed;inset:0;background:rgba(0,0,0,0.35);display:flex;align-items:center;justify-content:center;z-index:9999;color:#fff;font-size:16px;';
    o.textContent = text || '処理中…';
    document.body.appendChild(o);
    return o;
  }
  function hideOverlay(o) { if (o && o.parentNode) o.parentNode.removeChild(o); }

  function renderResult(groups) {
    const wrap = document.createElement('div');
    wrap.style.cssText = 'position:fixed;inset:5%;background:#fff;border-radius:8px;box-shadow:0 8px 30px rgba(0,0,0,.2);padding:16px;overflow:auto;z-index:10000;';
    wrap.innerHTML = '<h2 style="margin:0 0 10px;">重複候補の検出結果</h2>';
    const note = document.createElement('p');
    note.textContent = groups.length ? `候補グループ数: ${groups.length}` : '重複候補は見つかりませんでした。';
    wrap.appendChild(note);

    if (groups.length) {
      const table = document.createElement('table');
      table.style.cssText = 'width:100%;border-collapse:collapse;font-size:12px;';
      table.innerHTML = '<thead><tr><th style="border-bottom:2px solid #333;padding:8px;text-align:left;">ルール</th><th style="border-bottom:2px solid #333;padding:8px;text-align:left;">レコード</th></tr></thead>';
      const tbody = document.createElement('tbody');
      for (const g of groups) {
        const tr = document.createElement('tr');
        const tdRule = document.createElement('td');
        tdRule.style.borderBottom = '1px solid #eee';
        tdRule.textContent = g.rule + (g.score ? ` (${(g.score*100).toFixed(0)}%)` : '');
        const tdMembers = document.createElement('td');
        tdMembers.style.borderBottom = '1px solid #eee';
        tdMembers.appendChild(renderMembers(g.members));
        tr.appendChild(tdRule); tr.appendChild(tdMembers);
        tbody.appendChild(tr);
      }
      table.appendChild(tbody);
      wrap.appendChild(table);
    }

    const close = document.createElement('button');
    close.textContent = '閉じる';
    close.style.cssText = 'margin-top:12px; padding:6px 10px;';
    close.onclick = () => wrap.remove();
    wrap.appendChild(close);
    document.body.appendChild(wrap);
  }

  function renderMembers(members) {
    const appId = kintone.app.getId();
    const box = document.createElement('div');
    for (const r of members) {
      const a = document.createElement('a');
      a.href = `/k/${appId}/show#record=${r.$id.value}`;
      a.target = '_blank';
      a.textContent = `#${r.$id.value} ${r.name?.value || ''} / ${r.email?.value || ''} / ${r.tel?.value || ''} / ${r.dob?.value || ''}`;
      box.appendChild(a);
      box.appendChild(document.createElement('br'));
    }
    return box;
  }

})();
